// Cast Tab Keeper — Popup with keep-alive toggle
const KEY_ALLOW     = 'ctk_allowlist';
const KEY_CATALOG   = 'ctk_catalog';
const KEY_KEEPALIVE = 'ctk_keepalive';

const els = {
  host: document.getElementById('host'),
  status: document.getElementById('status'),
  allow: document.getElementById('allow'),
  block: document.getElementById('block'),
  remove: document.getElementById('remove'),
  openOptions: document.getElementById('openOptions'),
  keepToggle: document.getElementById('keepToggle'),
};

function normalizeDomain(v){
  if (typeof v !== 'string') return '';
  v = v.trim().toLowerCase().replace(/^https?:\/\//,'').replace(/\/.*$/,'').replace(/^www\./,'');
  return v;
}
function domainFromUrl(u){ try{ return normalizeDomain(new URL(u).hostname); } catch{ return ''; } }

async function getState(){
  const st = await chrome.storage.sync.get([KEY_ALLOW, KEY_CATALOG, KEY_KEEPALIVE]);
  return {
    allow: Array.isArray(st[KEY_ALLOW]) ? st[KEY_ALLOW] : [],
    catalog: Array.isArray(st[KEY_CATALOG]) ? st[KEY_CATALOG] : [],
    keep: (st[KEY_KEEPALIVE] && typeof st[KEY_KEEPALIVE] === 'object') ? st[KEY_KEEPALIVE] : { enabled: true, intervalMs: 300, corner: 'top-left' }
  };
}
async function setAllowlist(arr){
  arr = Array.from(new Set(arr));
  await chrome.storage.sync.set({ [KEY_ALLOW]: arr });
  await chrome.runtime.sendMessage({ type:'ctk:updateAllowlist', allowlist: arr }).catch(()=>{});
  return arr;
}
async function setCatalog(arr){
  arr = Array.from(new Set(arr));
  await chrome.storage.sync.set({ [KEY_CATALOG]: arr });
  return arr;
}

async function setKeepAliveEnabled(enabled){
  const st = await chrome.storage.sync.get(KEY_KEEPALIVE);
  const cfg = Object.assign({ enabled:true, intervalMs:300, corner:'top-left' }, st[KEY_KEEPALIVE] || {});
  cfg.enabled = !!enabled;
  await chrome.storage.sync.set({ [KEY_KEEPALIVE]: cfg });

  // notify only real pages (http/https/file)
  const tabs = await chrome.tabs.query({ url: ['http://*/*','https://*/*','file://*/*'] });
  for (const t of tabs){
    if (!t.url || !/^(https?|file):/i.test(t.url)) continue;
    chrome.tabs.sendMessage(t.id, { type:'ctk:keepalive:update', payload: cfg }).catch(()=>{});
  }
}

async function openOptions(){
  // Prefer options page if defined
  try {
    await chrome.runtime.openOptionsPage();
  } catch {
    chrome.tabs.create({ url: chrome.runtime.getURL('options.html') });
  }
}

async function init(){
  // current tab
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  const dom = domainFromUrl(tab?.url || '');
  els.host.textContent = dom || '—';

  const st = await getState();
  const isInCatalog = st.catalog.includes(dom);
  const isAllowed = st.allow.includes(dom);

  els.status.textContent = !dom ? 'No domain' :
    isInCatalog ? (isAllowed ? 'Allowed (smart-cast)' : 'Blocked (tab mirroring)')
                : 'Not in list';

  // buttons state
  els.allow.disabled = !dom;
  els.block.disabled = !dom;
  els.remove.disabled = !dom || !isInCatalog;

  // actions
  els.allow.onclick = async () => {
    if (!dom) return;
    let allow = st.allow.slice();
    let catalog = st.catalog.slice();
    if (!catalog.includes(dom)) catalog.push(dom);
    if (!allow.includes(dom)) allow.push(dom);
    await setCatalog(catalog);
    await setAllowlist(allow);
    window.close();
  };

  els.block.onclick = async () => {
    if (!dom) return;
    let allow = st.allow.filter(x => x !== dom);
    let catalog = st.catalog.slice();
    if (!catalog.includes(dom)) catalog.push(dom);
    await setCatalog(catalog);
    await setAllowlist(allow);
    window.close();
  };

  els.remove.onclick = async () => {
    if (!dom) return;
    let catalog = st.catalog.filter(x => x !== dom);
    let allow = st.allow.filter(x => x !== dom);
    await setCatalog(catalog);
    await setAllowlist(allow);
    window.close();
  };

  els.openOptions.onclick = openOptions;

  // keep-alive toggle (global)
  els.keepToggle.checked = !!st.keep.enabled;
  els.keepToggle.onchange = async () => {
    await setKeepAliveEnabled(els.keepToggle.checked);
  };
}

// silence "Receiving end does not exist" in popup (in case no content script)
window.addEventListener('unhandledrejection', (e)=>{
  const msg = String(e.reason || '');
  if (/Receiving end does not exist/i.test(msg)) e.preventDefault();
});

init();
