// Cast Tab Keeper — options with catalog, themes, and keep-alive controls.
const KEY_ALLOW     = 'ctk_allowlist';
const KEY_CATALOG   = 'ctk_catalog';
const KEY_THEME     = 'ctk_theme';      // 'blue' | 'green' | 'red'
const KEY_KEEPALIVE = 'ctk_keepalive';  // { enabled, intervalMs, corner }

const DEFAULT_ALLOWLIST = [
  "youtube.com",
  "music.youtube.com",
  "m.youtube.com",
  "youtube-nocookie.com"
];
const DEFAULT_KEEPALIVE = { enabled: true, intervalMs: 300, corner: 'top-left' };

const els = {
  domain: document.getElementById('domain'),
  add: document.getElementById('add'),
  addCurrent: document.getElementById('addCurrent'),
  list: document.getElementById('list'),
  empty: document.getElementById('empty'),
  count: document.getElementById('count'),
  presetYt: document.getElementById('presetYt'),
  exportBtn: document.getElementById('exportBtn'),
  importBtn: document.getElementById('importBtn'),
  importFile: document.getElementById('importFile'),
  resetBtn: document.getElementById('resetBtn'),
  syncState: document.getElementById('syncState'),
  themeSelect: document.getElementById('themeSelect'),
  themeDot: document.getElementById('themeDot'),

  // keep-alive
  keepEnabled: document.getElementById('keepEnabled'),
  keepInterval: document.getElementById('keepInterval'),
  keepCorner: document.getElementById('keepCorner'),
  keepSave: document.getElementById('keepSave'),
};

window.addEventListener('unhandledrejection', (e) => {
  const msg = String(e.reason || '');
  if (/Receiving end does not exist/i.test(msg)) {
    e.preventDefault(); // не засоряем консоль
  }
});

// ---------- utils ----------
function normalizeDomain(v) {
  if (typeof v !== 'string') return '';
  v = v.trim().toLowerCase();
  v = v.replace(/^https?:\/\//,'').replace(/\/.*$/,'');
  v = v.replace(/^www\./,'');
  return v;
}
function isValidHost(v) {
  return /^[a-z0-9.-]+$/.test(v) && !v.startsWith('.') && !v.endsWith('.') && v.length <= 253;
}
function uniqSorted(arr) {
  const out = [];
  for (let v of arr) {
    v = normalizeDomain(v);
    if (!v) continue;
    if (!out.includes(v)) out.push(v);
  }
  out.sort((a,b)=>a.localeCompare(b));
  return out;
}
function domainFromUrl(u) {
  try { return normalizeDomain(new URL(u).hostname); } catch { return ''; }
}

// ---------- storage ----------
async function getState() {
  const data = await chrome.storage.sync.get([KEY_ALLOW, KEY_CATALOG]);
  const allow = Array.isArray(data[KEY_ALLOW]) ? data[KEY_ALLOW] : [];
  const catalog = Array.isArray(data[KEY_CATALOG]) ? data[KEY_CATALOG] : [];
  return { allow: uniqSorted(allow), catalog: uniqSorted(catalog) };
}
async function setAllowlist(allow) {
  allow = uniqSorted(allow);
  await chrome.storage.sync.set({ [KEY_ALLOW]: allow });
  await chrome.runtime.sendMessage({ type: 'ctk:updateAllowlist', allowlist: allow }).catch(()=>{});
  return allow;
}
async function setCatalog(catalog) {
  catalog = uniqSorted(catalog);
  await chrome.storage.sync.set({ [KEY_CATALOG]: catalog });
  return catalog;
}

// ---------- theme ----------
function applyTheme(theme) {
  const root = document.documentElement;
  root.classList.remove('theme-blue','theme-green','theme-red');
  const cls = theme === 'green' ? 'theme-green' : theme === 'red' ? 'theme-red' : 'theme-blue';
  root.classList.add(cls);
  const accent = getComputedStyle(root).getPropertyValue('--accent').trim() || '#2563eb';
  els.themeDot.style.background = accent;
}
async function initTheme() {
  const data = await chrome.storage.sync.get(KEY_THEME);
  const theme = (data[KEY_THEME] || 'blue');
  els.themeSelect.value = theme;
  applyTheme(theme);
}
els.themeSelect.addEventListener('change', async ()=>{
  const val = els.themeSelect.value;
  await chrome.storage.sync.set({ [KEY_THEME]: val });
  applyTheme(val);
});

// ---------- keep-alive ----------
async function loadKeepAlive() {
  const data = await chrome.storage.sync.get(KEY_KEEPALIVE);
  const cfg = { ...DEFAULT_KEEPALIVE, ...(data[KEY_KEEPALIVE] || {}) };
  els.keepEnabled.checked = !!cfg.enabled;
  els.keepInterval.value = String(cfg.intervalMs);
  els.keepCorner.value = cfg.corner;
  return cfg;
}

// Patched: send only to tabs where content script can exist; suppress "receiving end" errors
async function saveKeepAlive(cfg) {
  await chrome.storage.sync.set({ [KEY_KEEPALIVE]: cfg });

  // Только реальные страницы (без chrome://, edge:// и т.п.)
  const tabs = await chrome.tabs.query({
    url: ['http://*/*', 'https://*/*', 'file://*/*']
  });

  for (const t of tabs) {
    try {
      if (!t.url || !/^(https?|file):/i.test(t.url)) continue;
      await chrome.tabs.sendMessage(t.id, { type: 'ctk:keepalive:update', payload: cfg })
        .catch(() => {}); // подстрахуемся и тут
    } catch (_) { /* ignore */ }
  }
}

els.keepSave.addEventListener('click', async ()=>{
  let interval = parseInt(els.keepInterval.value, 10);
  if (Number.isNaN(interval)) interval = DEFAULT_KEEPALIVE.intervalMs;
  interval = Math.min(2000, Math.max(120, interval));
  const cfg = {
    enabled: !!els.keepEnabled.checked,
    intervalMs: interval,
    corner: els.keepCorner.value
  };
  await saveKeepAlive(cfg);
  els.keepSave.disabled = true;
  setTimeout(()=>{ els.keepSave.disabled = false; }, 400);
});

// ---------- render ----------
function domainRow(dom, isAllowed) {
  const li = document.createElement('li');

  const left = document.createElement('div');
  const title = document.createElement('div');
  title.className = 'dom';
  title.textContent = dom;
  const sub = document.createElement('div');
  sub.className = 'muted';
  sub.textContent = isAllowed ? 'Allowed (smart-cast)' : 'Blocked (tab mirroring)';
  left.appendChild(title);
  left.appendChild(sub);

  const right = document.createElement('div');
  right.className = 'row';

  const cb = document.createElement('input');
  cb.type = 'checkbox';
  cb.checked = isAllowed;
  cb.title = isAllowed ? 'Uncheck to block (keep tab mirroring)' : 'Check to allow smart-cast';
  cb.addEventListener('change', async () => {
    let { allow } = await getState();
    if (cb.checked) {
      if (!allow.includes(dom)) allow.push(dom);
    } else {
      allow = allow.filter(x => x !== dom);
    }
    allow = await setAllowlist(allow);
    sub.textContent = cb.checked ? 'Allowed (smart-cast)' : 'Blocked (tab mirroring)';
  });

  const del = document.createElement('button');
  del.textContent = 'Remove';
  del.addEventListener('click', async () => {
    let { allow, catalog } = await getState();
    catalog = catalog.filter(x => x !== dom);
    allow = allow.filter(x => x !== dom);
    await setCatalog(catalog);
    await setAllowlist(allow);
    renderList();
  });

  right.appendChild(cb);
  right.appendChild(del);

  li.appendChild(left);
  li.appendChild(right);
  return li;
}

async function renderList() {
  const { allow, catalog } = await getState();
  els.list.innerHTML = '';
  const rows = catalog.map(dom => domainRow(dom, allow.includes(dom)));
  rows.forEach(r => els.list.appendChild(r));
  els.count.textContent = String(catalog.length);
  els.empty.style.display = catalog.length ? 'none' : 'block';
}

// ---------- actions ----------
els.add.addEventListener('click', async ()=>{
  const d = normalizeDomain(els.domain.value);
  if (!d || !isValidHost(d)) return;
  let { catalog } = await getState();
  if (!catalog.includes(d)) {
    catalog.push(d);
    await setCatalog(catalog);
  }
  els.domain.value = '';
  renderList();
});

els.domain.addEventListener('keydown', (e)=>{
  if (e.key === 'Enter') els.add.click();
});

els.addCurrent.addEventListener('click', async ()=>{
  const [tab] = await chrome.tabs.query({active: true, currentWindow: true});
  const dom = domainFromUrl(tab?.url || '');
  if (!dom || !isValidHost(dom)) return;
  let { catalog } = await getState();
  if (!catalog.includes(dom)) {
    catalog.push(dom);
    await setCatalog(catalog);
  }
  renderList();
});

els.presetYt.addEventListener('click', async ()=>{
  let { catalog, allow } = await getState();
  const merged = uniqSorted([...catalog, ...DEFAULT_ALLOWLIST]);
  await setCatalog(merged);
  const allowMerged = uniqSorted([...allow, ...DEFAULT_ALLOWLIST]);
  await setAllowlist(allowMerged);
  renderList();
});

els.exportBtn.addEventListener('click', async ()=>{
  const st = await chrome.storage.sync.get([KEY_ALLOW, KEY_CATALOG, KEY_THEME, KEY_KEEPALIVE]);
  const payload = {
    version: 1,
    exportedAt: new Date().toISOString(),
    allowlist: Array.isArray(st[KEY_ALLOW]) ? st[KEY_ALLOW] : [],
    catalog: Array.isArray(st[KEY_CATALOG]) ? st[KEY_CATALOG] : [],
    theme: st[KEY_THEME] || 'blue',
    keepalive: { ...DEFAULT_KEEPALIVE, ...(st[KEY_KEEPALIVE] || {}) }
  };
  const blob = new Blob([JSON.stringify(payload, null, 2)], {type:'application/json'});
  const url = URL.createObjectURL(blob);
  const a = document.createElement('a');
  a.href = url;
  a.download = 'cast-tab-keeper-settings.json';
  document.body.appendChild(a);
  a.click();
  a.remove();
  URL.revokeObjectURL(url);
});

els.importBtn.addEventListener('click', ()=> els.importFile.click());
els.importFile.addEventListener('change', async (e)=>{
  const file = e.target.files?.[0];
  if (!file) return;
  try {
    const text = await file.text();
    const json = JSON.parse(text);
    const importedAllow = Array.isArray(json.allowlist) ? json.allowlist : [];
    const importedCatalog = Array.isArray(json.catalog) ? json.catalog : [];
    const importedTheme = typeof json.theme === 'string' ? json.theme : undefined;
    const importedKeep = (json.keepalive && typeof json.keepalive === 'object') ? json.keepalive : undefined;

    let { allow, catalog } = await getState();
    allow = uniqSorted([...allow, ...importedAllow]);
    catalog = uniqSorted([...catalog, ...importedCatalog]);

    await setCatalog(catalog);
    await setAllowlist(allow);

    if (importedTheme) {
      await chrome.storage.sync.set({ [KEY_THEME]: importedTheme });
      applyTheme(importedTheme);
      els.themeSelect.value = importedTheme;
    }
    if (importedKeep) {
      const cfg = {
        enabled: typeof importedKeep.enabled === 'boolean' ? importedKeep.enabled : DEFAULT_KEEPALIVE.enabled,
        intervalMs: Number.isFinite(importedKeep.intervalMs) ? importedKeep.intervalMs : DEFAULT_KEEPALIVE.intervalMs,
        corner: importedKeep.corner || DEFAULT_KEEPALIVE.corner
      };
      await saveKeepAlive(cfg);
      els.keepEnabled.checked = cfg.enabled;
      els.keepInterval.value = String(cfg.intervalMs);
      els.keepCorner.value = cfg.corner;
    }
    renderList();
  } catch (err) {
    console.error('Import failed:', err);
  } finally {
    els.importFile.value = '';
  }
});

els.resetBtn.addEventListener('click', async ()=>{
  await setCatalog(DEFAULT_ALLOWLIST.slice());
  await setAllowlist(DEFAULT_ALLOWLIST.slice());
  await chrome.storage.sync.set({ [KEY_THEME]: 'blue', [KEY_KEEPALIVE]: DEFAULT_KEEPALIVE });
  applyTheme('blue');
  els.themeSelect.value = 'blue';
  els.keepEnabled.checked = DEFAULT_KEEPALIVE.enabled;
  els.keepInterval.value = String(DEFAULT_KEEPALIVE.intervalMs);
  els.keepCorner.value = DEFAULT_KEEPALIVE.corner;
  await saveKeepAlive(DEFAULT_KEEPALIVE);
  renderList();
});

// ---------- init ----------
(async function init(){
  els.syncState.textContent = 'sync';
  await initTheme();
  await loadKeepAlive();
  await renderList();
})();
