// Cast Tab Keeper — tiny keep-alive for tab mirroring.
// Strategy: toggle a 1x1 fixed pixel with near-transparent alpha at ~3 Hz
(() => {
  const DEFAULTS = { enabled: true, intervalMs: 300, corner: 'top-left' };
  const KEY = 'ctk_keepalive';
  let state = { ...DEFAULTS };
  let el = null, timer = null, flip = false;

  const corners = {
    'top-left':   { top: '0', left: '0', right: 'auto', bottom: 'auto' },
    'top-right':  { top: '0', right: '0', left: 'auto', bottom: 'auto' },
    'bottom-left':{ bottom:'0', left: '0', right: 'auto', top: 'auto' },
    'bottom-right':{ bottom:'0', right:'0', left: 'auto', top: 'auto' },
  };

  function setCorner(node, name) {
    node.style.top = node.style.right = node.style.bottom = node.style.left = 'auto';
    Object.assign(node.style, corners[name] || corners['top-left']);
  }

  function makePixel() {
    const d = document.createElement('div');
    Object.assign(d.style, {
      position: 'fixed',
      width: '1px',
      height: '1px',
      zIndex: '2147483647',
      pointerEvents: 'none',
      backgroundColor: 'rgba(0,0,0,0.003)',
      mixBlendMode: 'normal'
    });
    setCorner(d, state.corner);
    (document.documentElement || document.body).appendChild(d);
    return d;
  }

  function start() {
    if (!state.enabled || timer) return;
    if (!el) el = makePixel();
    timer = setInterval(() => {
      flip = !flip;
      el.style.backgroundColor = flip ? 'rgba(0,0,0,0.004)' : 'rgba(0,0,0,0.003)';
    }, Math.max(120, state.intervalMs | 0));
  }

  function stop() {
    if (timer) { clearInterval(timer); timer = null; }
    if (el && el.parentNode) { el.parentNode.removeChild(el); el = null; }
  }

  function apply(newState) {
    const prevEnabled = state.enabled;
    state = { ...state, ...newState };
    if (el) setCorner(el, state.corner);
    if (state.enabled && !prevEnabled) start();
    if (!state.enabled && prevEnabled) stop();
  }

  // initial load from storage
  try {
    chrome.storage?.sync?.get(KEY).then(data => {
      apply(data?.[KEY] || DEFAULTS);
      if (state.enabled) start();
    }).catch(() => { apply(DEFAULTS); start(); });
  } catch {
    apply(DEFAULTS); start();
  }

  // live updates from options
  try {
    chrome.runtime.onMessage.addListener((msg) => {
      if (msg?.type === 'ctk:keepalive:update') {
        apply(msg.payload || {});
      }
    });
  } catch {}

  window.addEventListener('pagehide', stop);
})();
