// content.js

// 1) Inject page context patch (so we can redefine window/navigator props)
(function inject() {
  const s = document.createElement('script');
  s.src = chrome.runtime.getURL('page_patch.js');
  s.type = 'text/javascript';
  // run as early as possible
  (document.documentElement || document.head || document.body).appendChild(s);
  s.remove();
})();

// 2) Hide/remove cast buttons visible in DOM
const hideStyle = document.createElement('style');
hideStyle.textContent = `
  button[is="google-cast-button"], google-cast-button {
    display: none !important;
    visibility: hidden !important;
    pointer-events: none !important;
  }
`;
(document.head || document.documentElement).appendChild(hideStyle);

function hardenMedia(root = document) {
  const nodes = root.querySelectorAll('video,audio');
  for (const el of nodes) {
    try {
      el.disableRemotePlayback = true;
      el.setAttribute('disableRemotePlayback', '');
    } catch (_) {}
  }
}

function removeCastButtons(root = document) {
  root.querySelectorAll('button[is="google-cast-button"], google-cast-button')
      .forEach(n => n.remove());
}

// Initial pass
hardenMedia();
removeCastButtons();

// Observe DOM and Shadow DOM
const mo = new MutationObserver(muts => {
  for (const m of muts) {
    for (const n of m.addedNodes) {
      if (n.nodeType !== 1) continue;
      if (n.matches) {
        if (n.matches('video,audio')) hardenMedia(n.parentNode || document);
        if (n.matches('button[is="google-cast-button"], google-cast-button')) n.remove();
      }
      hardenMedia(n);
      removeCastButtons(n);
    }
  }
});
mo.observe(document.documentElement, { childList: true, subtree: true });

// Patch ShadowRoot.observe via attachShadow
const _attach = Element.prototype.attachShadow;
Element.prototype.attachShadow = function (init) {
  const sr = _attach.call(this, init);
  try {
    const mo2 = new MutationObserver(muts => {
      for (const m of muts) {
        for (const n of m.addedNodes) {
          if (n.nodeType !== 1) continue;
          if (n.matches && n.matches('video,audio')) hardenMedia(n.parentNode || sr);
          hardenMedia(n);
          removeCastButtons(n);
        }
      }
    });
    mo2.observe(sr, { childList: true, subtree: true });
    hardenMedia(sr);
    removeCastButtons(sr);
  } catch (_) {}
  return sr;
};
