// background.js (MV3, module)

const DNR = chrome.declarativeNetRequest;
const STORAGE_KEY = 'ctk_allowlist';
const DYNAMIC_RULE_BASE_ID = 1000; // reserve id range for our allow rules
const TARGETS = [
  "www.gstatic.com/cv/js/sender/v1/cast_sender.js",
  "www.gstatic.com/cast/sdk/"
];

const DEFAULT_ALLOWLIST = [
  "youtube.com",
  "music.youtube.com",
  "m.youtube.com",
  "youtube-nocookie.com"
];

async function getAllowlist() {
  const { [STORAGE_KEY]: list } = await chrome.storage.sync.get(STORAGE_KEY);
  return Array.isArray(list) ? list : DEFAULT_ALLOWLIST.slice();
}

function makeAllowRules(allowlist) {
  let id = DYNAMIC_RULE_BASE_ID;
  const rules = [];
  for (const domain of allowlist) {
    for (const target of TARGETS) {
      rules.push({
        id: id++,
        priority: 100,
        action: { type: "allow" },
        condition: {
          // initiatorDomains: domain that initiates the request (top-level or frame)
          initiatorDomains: [domain],
          urlFilter: target,
          resourceTypes: ["script"]
        }
      });
    }
  }
  return rules;
}

async function applyAllowRules() {
  const allowlist = await getAllowlist();
  const newRules = makeAllowRules(allowlist);

  // Remove previous dynamic rules in our id range
  const existing = await DNR.getDynamicRules();
  const toRemove = existing
    .filter(r => r.id >= DYNAMIC_RULE_BASE_ID)
    .map(r => r.id);

  await DNR.updateDynamicRules({
    removeRuleIds: toRemove,
    addRules: newRules
  });
}

chrome.runtime.onInstalled.addListener(async (details) => {
  if (details.reason === 'install') {
    await chrome.storage.sync.set({ [STORAGE_KEY]: DEFAULT_ALLOWLIST });
  }
  await applyAllowRules();
});

chrome.runtime.onStartup.addListener(applyAllowRules);

// Receive updates from options page
chrome.runtime.onMessage.addListener(async (msg, sender, sendResponse) => {
  if (msg && msg.type === 'ctk:updateAllowlist' && Array.isArray(msg.allowlist)) {
    await chrome.storage.sync.set({ [STORAGE_KEY]: msg.allowlist });
    await applyAllowRules();
    sendResponse({ ok: true });
  }
});

chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  // Ничего «тяжёлого» тут не нужно — просто подтверждаем приём
  if (msg && msg.type === 'ctk:updateAllowlist') {
    // если у тебя уже есть своя логика обновления DNR-правил — оставь её здесь
    sendResponse({ ok: true });
    return true; // держим порт открытым, если где-то будет async
  }
  return false;
});